#!/usr/bin/perl -w
# Script added to YaBB to be able to upload images instead of just linking to
# them.
#
# YaBB by default allows you to link to images, but that supposes that you
# have web space and the know-how to put images onto the web before linking to
# them... Or that you will only link to images from other web sites. With this
# script, a user can upload an image to the site where the YaBB is being hosted,
# and the image will be stored on it and linked to there from their message.
#
# The script opens in a pop-up which displays a plain HTML file that contains
# a form used to choose the file.
#
# Once the user has chosen his file and clicked OK, the script will check that
# the file is not larger than a certain limit (set to 200KB by default) and that
# it is indeed an image (by mime-type), and will then save the image to a path
# specified in the script.
#
# Then, the pop-up is closed, and either the [img] tag for the image or an
# error message will be put in the message text area.
#
# This script is Copyright (c) 2003 by Jean-Sbastien Guay
# jean_seb@videotron.ca  --  http://whitestar02.webhop.org/
#
use strict;
use CGI;
use CGI::Carp 'fatalsToBrowser';

sub cleanupTemp {
    # This is only necessary when the filehandle that CGI.pm gives us to read the
    # uploaded file is not closed properly. If it is, then no temp files are left
    # in the current directory.
    foreach my $file (`ls CGItemp*`) {
        `rm $file`;
    }
}

# Puts any text into the text field on the post page.
sub printToTextField {
    my $text = shift;

    print qq[<script type="text/javascript" language="javascript">
                 var text = '$text';
                 if (opener.document.postmodify.message.caretPos)
                     opener.document.postmodify.message.caretPos.text =
                         text + opener.document.postmodify.message.caretPos.text;
                 else opener.document.postmodify.message.value += text;
                 opener.document.postmodify.message.focus();
                 window.close();
             </script>];
}

# Puts an error message into the text field (for the user) and in the web
# server's logs (with die).
sub errorMessage {
    my $message = shift;

    printToTextField($message);
    die $message;
}

my $q = new CGI;
my $MAXLENGTH = 1024 * 200;       # Max 200 KB uploads
$CGI::POST_MAX = $MAXLENGTH;      # Limit the size of POSTs at the source.
my $PATH = '../../images/BoardImages';      # Where to store the images
my $HTML_PATH = 'http://mysite/images/BoardImages';    # Client-side equivalent

print qq[Content-Type: text/html\n\n];

if ($q->param('submitted')) {
    # Get the file name of the uploaded file
    my $filename = $q->param('Data');
    if (!$filename) {
        errorMessage "No image uploaded.";
    }

    # Extract just the name (blabla.jpg)
    $filename =~ m/[\\\/]([^\\\/]+)$/;
    my $basename = $1;
    $filename = "$PATH/$1";
    my $html_filename = "$HTML_PATH/$1";
    print $filename . "<br>\n";

    # Check if the file is an image
    my $type = $q->uploadInfo($q->param('Data'))->{'Content-Type'};
    print "Content-type: $type";
    if ($type ne "image/jpg" &&
        $type ne "image/pjpg" &&
        $type ne "image/jpeg" &&
        $type ne "image/pjpeg" &&
        $type ne "image/gif" &&
        $type ne "image/png") {
            errorMessage "Image is not a web-supported type: $basename type: $type";
    }

    # Check if the data is OK, and only then open the other file for writing,
    # or else we'll create an empty file if the data is not good...
    my $fh = $q->upload('Data')
        or errorMessage "File was not correctly uploaded, please try again: $!";

    # I do not check for overwrites at all. That means that if there was an old
    # file that had the same name as the one that's being uploaded, then the image
    # in the old post will have a different image (unless the images were
    # identical AND had the same name, of course).
    open FILE, '>', $filename
        or close $fh && errorMessage "Could not open $filename for writing: $!";

    # Make sure we're in binary mode for read and write.
    binmode FILE;
    binmode $fh;

    my $len = 0;
    # Write the data to the file
    while (<$fh>) {
        $len += length;
        print FILE;
    }

    close FILE;
    close $fh;

    # Check the length of the file...
    # We set a max POST size before, but it doesn't hurt to double-check
    if ($len > $MAXLENGTH) {
        # If it's too large, delete it
        unlink $filename;
        errorMessage "File $basename exceeds " . $MAXLENGTH / 1024 .
            " KB, upload aborted.";
    }

    # Put the code to display the image into the text field on the post page,
    # and close this pop-up window.
    printToTextField('[img]' . $html_filename . '[/img]');

    # Cleanup temp files left by CGI.pm when uploading a file
    # Was needed before I found out how to close a file AND call errorMessage
    # (hint : close $fh && errorMessage "blah";)
    #cleanupTemp;
}
else {
    # Display the page with the form to let the user pick a file
    open FILE, "upload.html"
        or errorMessage "Could not open upload.html: $!";

    print <FILE>;

    close FILE
        or errorMessage "Could not close upload.html: $!";
}